﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/deadline/DeadlineRequest.h>
#include <aws/deadline/Deadline_EXPORTS.h>
#include <aws/deadline/model/FleetConfiguration.h>
#include <aws/deadline/model/HostConfiguration.h>

#include <utility>

namespace Aws {
namespace deadline {
namespace Model {

/**
 */
class CreateFleetRequest : public DeadlineRequest {
 public:
  AWS_DEADLINE_API CreateFleetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateFleet"; }

  AWS_DEADLINE_API Aws::String SerializePayload() const override;

  AWS_DEADLINE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The unique token which the server uses to recognize retries of the same
   * request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateFleetRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The farm ID of the farm to connect to the fleet.</p>
   */
  inline const Aws::String& GetFarmId() const { return m_farmId; }
  inline bool FarmIdHasBeenSet() const { return m_farmIdHasBeenSet; }
  template <typename FarmIdT = Aws::String>
  void SetFarmId(FarmIdT&& value) {
    m_farmIdHasBeenSet = true;
    m_farmId = std::forward<FarmIdT>(value);
  }
  template <typename FarmIdT = Aws::String>
  CreateFleetRequest& WithFarmId(FarmIdT&& value) {
    SetFarmId(std::forward<FarmIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The display name of the fleet.</p>  <p>This field can store any
   * content. Escape or encode this content before displaying it on a webpage or any
   * other system that might interpret the content of this field.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  CreateFleetRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the fleet.</p>  <p>This field can store any
   * content. Escape or encode this content before displaying it on a webpage or any
   * other system that might interpret the content of this field.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateFleetRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IAM role ARN for the role that the fleet's workers will use.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateFleetRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The minimum number of workers for the fleet.</p>
   */
  inline int GetMinWorkerCount() const { return m_minWorkerCount; }
  inline bool MinWorkerCountHasBeenSet() const { return m_minWorkerCountHasBeenSet; }
  inline void SetMinWorkerCount(int value) {
    m_minWorkerCountHasBeenSet = true;
    m_minWorkerCount = value;
  }
  inline CreateFleetRequest& WithMinWorkerCount(int value) {
    SetMinWorkerCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of workers for the fleet.</p> <p>Deadline Cloud limits the
   * number of workers to less than or equal to the fleet's maximum worker count. The
   * service maintains eventual consistency for the worker count. If you make
   * multiple rapid calls to <code>CreateWorker</code> before the field updates, you
   * might exceed your fleet's maximum worker count. For example, if your
   * <code>maxWorkerCount</code> is 10 and you currently have 9 workers, making two
   * quick <code>CreateWorker</code> calls might successfully create 2 workers
   * instead of 1, resulting in 11 total workers.</p>
   */
  inline int GetMaxWorkerCount() const { return m_maxWorkerCount; }
  inline bool MaxWorkerCountHasBeenSet() const { return m_maxWorkerCountHasBeenSet; }
  inline void SetMaxWorkerCount(int value) {
    m_maxWorkerCountHasBeenSet = true;
    m_maxWorkerCount = value;
  }
  inline CreateFleetRequest& WithMaxWorkerCount(int value) {
    SetMaxWorkerCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration settings for the fleet. Customer managed fleets are
   * self-managed. Service managed Amazon EC2 fleets are managed by Deadline
   * Cloud.</p>
   */
  inline const FleetConfiguration& GetConfiguration() const { return m_configuration; }
  inline bool ConfigurationHasBeenSet() const { return m_configurationHasBeenSet; }
  template <typename ConfigurationT = FleetConfiguration>
  void SetConfiguration(ConfigurationT&& value) {
    m_configurationHasBeenSet = true;
    m_configuration = std::forward<ConfigurationT>(value);
  }
  template <typename ConfigurationT = FleetConfiguration>
  CreateFleetRequest& WithConfiguration(ConfigurationT&& value) {
    SetConfiguration(std::forward<ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Each tag consists of a tag key and a tag value. Tag keys and values are both
   * required, but tag values can be empty strings.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateFleetRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateFleetRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides a script that runs as a worker is starting up that you can use to
   * provide additional configuration for workers in your fleet.</p>
   */
  inline const HostConfiguration& GetHostConfiguration() const { return m_hostConfiguration; }
  inline bool HostConfigurationHasBeenSet() const { return m_hostConfigurationHasBeenSet; }
  template <typename HostConfigurationT = HostConfiguration>
  void SetHostConfiguration(HostConfigurationT&& value) {
    m_hostConfigurationHasBeenSet = true;
    m_hostConfiguration = std::forward<HostConfigurationT>(value);
  }
  template <typename HostConfigurationT = HostConfiguration>
  CreateFleetRequest& WithHostConfiguration(HostConfigurationT&& value) {
    SetHostConfiguration(std::forward<HostConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_farmId;
  bool m_farmIdHasBeenSet = false;

  Aws::String m_displayName;
  bool m_displayNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  int m_minWorkerCount{0};
  bool m_minWorkerCountHasBeenSet = false;

  int m_maxWorkerCount{0};
  bool m_maxWorkerCountHasBeenSet = false;

  FleetConfiguration m_configuration;
  bool m_configurationHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  HostConfiguration m_hostConfiguration;
  bool m_hostConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace deadline
}  // namespace Aws
