use core:lang;

// Implement support for Int[] instead of Array<Int>
SrcName arrayType(SrcName t) on Compiler {
	SrcName name(t.pos);
	name.add("core");

	RecPart part("Array");
	part.add(t);

	name.add(part);
	name;
}

Expr arrayInit(Block block, SrcName type, Array<Expr> list) on Compiler {
	if (list.empty) {
		// Empty, just create the array.
		namedExpr(block, arrayType(type), Actuals());
	} else if (list.count == 1) {
		// One element, just use the constructor directly.
		Actuals params;
		params.add(NumLiteral(type.pos, 1));
		params.add(list[0]);
		namedExpr(block, arrayType(type), params);
	} else {
		// General case:
		ExprBlock b(type.pos, block);

		SStr arrayName("tmp", type.pos);
		Var array(b, arrayName, namedExpr(b, arrayType(type), Actuals()));
		b.add(array);

		LocalVarAccess varAccess(type.pos, array.var);
		NumLiteral size(type.pos, list.count.long);
		b.add(namedExpr(b, SStr("reserve", type.pos), varAccess, Actuals(size)));

		SStr push("push", type.pos);
		for (Nat i = 0; i < list.count; i++) {
			Actuals params;
			params.add(list[i]);
			b.add(namedExpr(b, push, varAccess, params));
		}

		b.add(varAccess);
		b;
	}
}

class ArrayInit extends Expr {
	Array<Expr> values;

	ExprBlock block; // TODO: Replace with direct ref to parent?

	init(SrcPos pos, Block parent, Array<Expr> values) {
		init(pos) {
			values = values;
			block = ExprBlock(pos, parent);
		}
	}

	ExprResult result() {
		if (values.any) {
			// TODO: Use the common function that takes Expr parameters?
			ExprResult r = values[0].result;
			for (Nat i = 0; i < values.count; i++) {
				r = common(r, values[i].result);
			}
			wrapArray(r.type);
		} else {
			Value();
		}
	}

	Int castPenalty(Value to) {
		if (to.ref) {
			-1;
		} else if (isArray(to)) {
			if (acceptable(unwrapArray(to))) {
				1;
			} else {
				-1;
			}
		} else {
			-1;
		}
	}

	void code(CodeGen g, CodeResult r) {
		Value v = unwrapArray(r.type);
		if (type = v.type) {
			Expr e = arrayInit(block, SrcName(type.path(), SrcPos()), values);
			e.code(g, r);
		} else {
			throw SyntaxError(pos, "Can not figure out the type of the array. Please use the syntax T:[...] instead.");
		}
	}

	// Is the type 't' acceptable for all values?
	Bool acceptable(Value t) {
		Bool ok = true;
		for (Nat i = 0; i < values.count; i++) {
			ok = ok & lang:bs:castable(values[i], t, block.scope);
		}

		ok;
	}

	Str toS() {
		StrBuf b;
		b.add("[");

		for (Nat i = 0; i < values.count; i++) {
			if (i > 0)
				b.add(", ");
			b.add(values[i].toS);
		}

		b.add("]");
		b.toS;
	}
}

