/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.store.remote.utils.cache;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiFunction;
import java.util.function.Predicate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.common.cache.RemovalListener;
import org.opensearch.common.cache.RemovalNotification;
import org.opensearch.common.cache.RemovalReason;
import org.opensearch.common.cache.Weigher;
import org.opensearch.index.store.remote.utils.cache.RefCountedCache;
import org.opensearch.index.store.remote.utils.cache.stats.FileStatsCounter;
import org.opensearch.index.store.remote.utils.cache.stats.IRefCountedCacheStats;
import org.opensearch.index.store.remote.utils.cache.stats.StatsCounter;

class LRUCache<K, V>
implements RefCountedCache<K, V> {
    private static final Logger logger = LogManager.getLogger(LRUCache.class);
    private final long capacity;
    private final HashMap<K, Node<K, V>> data;
    private final LinkedHashMap<K, Node<K, V>> lru;
    private final RemovalListener<K, V> listener;
    private final Weigher<V> weigher;
    private final StatsCounter<K, V> statsCounter;
    private final ReentrantLock lock;

    public LRUCache(long capacity, RemovalListener<K, V> listener, Weigher<V> weigher) {
        this.capacity = capacity;
        this.listener = listener;
        this.weigher = weigher;
        this.data = new HashMap();
        this.lru = new LinkedHashMap();
        this.lock = new ReentrantLock();
        this.statsCounter = new FileStatsCounter();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V get(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node == null) {
                this.statsCounter.recordMisses(key, 1);
                V v = null;
                return v;
            }
            this.incRef(key);
            this.statsCounter.recordHits(key, node.value, node.pinned, 1);
            Object v = node.value;
            return v;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V put(K key, V value) {
        Objects.requireNonNull(key);
        Objects.requireNonNull(value);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null) {
                Object oldValue = node.value;
                this.replaceNode(node, value);
                Object v = oldValue;
                return v;
            }
            this.addNode(key, false, value);
            V v = null;
            return v;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public V compute(K key, BiFunction<? super K, ? super V, ? extends V> remappingFunction) {
        Objects.requireNonNull(key);
        Objects.requireNonNull(remappingFunction);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node == null) {
                V newValue = remappingFunction.apply(key, null);
                if (newValue == null) {
                    V v = null;
                    return v;
                }
                this.addNode(key, false, newValue);
                this.statsCounter.recordMisses(key, 1);
                V v = newValue;
                return v;
            }
            V newValue = remappingFunction.apply(key, node.value);
            if (newValue == null) {
                this.removeNode(key);
                V v = null;
                return v;
            }
            this.statsCounter.recordHits(key, node.value, node.pinned, 1);
            this.replaceNode(node, newValue);
            V v = newValue;
            return v;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void remove(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            this.removeNode(key);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void clear() {
        this.lock.lock();
        try {
            this.lru.clear();
            Iterator<Node<K, V>> iterator = this.data.values().iterator();
            while (iterator.hasNext()) {
                Node<K, V> node = iterator.next();
                iterator.remove();
                this.statsCounter.recordRemoval(node.value, node.pinned, node.weight);
                this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
            }
            this.statsCounter.resetUsage();
            this.statsCounter.resetActiveUsage();
            this.statsCounter.resetPinnedUsage();
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public long size() {
        return this.data.size();
    }

    @Override
    public void incRef(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null) {
                if (node.refCount == 0) {
                    this.statsCounter.recordActiveUsage(node.value, node.weight, node.pinned, false);
                }
                if (node.evictable()) {
                    this.lru.remove(node.key);
                }
                ++node.refCount;
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void decRef(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null && node.refCount > 0) {
                --node.refCount;
                if (node.evictable()) {
                    this.lru.put(node.key, node);
                }
                if (node.refCount == 0) {
                    this.statsCounter.recordActiveUsage(node.value, node.weight, node.pinned, true);
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void pin(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null) {
                if (!node.pinned) {
                    this.statsCounter.recordPinnedUsage(node.value, node.weight, false);
                }
                if (node.evictable()) {
                    this.lru.remove(node.key, node);
                }
                node.pinned = true;
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void unpin(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null && node.pinned) {
                node.pinned = false;
                if (node.evictable()) {
                    this.lru.put(node.key, node);
                }
                this.statsCounter.recordPinnedUsage(node.value, node.weight, true);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Integer getRef(K key) {
        Objects.requireNonNull(key);
        this.lock.lock();
        try {
            Node<K, V> node = this.data.get(key);
            if (node != null) {
                Integer n = node.refCount;
                return n;
            }
            Integer n = null;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long prune(Predicate<K> keyPredicate) {
        long sum = 0L;
        this.lock.lock();
        try {
            Iterator<Node<K, V>> iterator = this.lru.values().iterator();
            while (iterator.hasNext()) {
                Node<K, V> node = iterator.next();
                if (keyPredicate != null && !keyPredicate.test(node.key)) continue;
                iterator.remove();
                this.data.remove(node.key, node);
                sum += node.weight;
                this.statsCounter.recordRemoval(node.value, node.pinned, node.weight);
                this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
            }
        }
        finally {
            this.lock.unlock();
        }
        return sum;
    }

    @Override
    public long usage() {
        this.lock.lock();
        try {
            long l = this.statsCounter.usage();
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public long activeUsage() {
        this.lock.lock();
        try {
            long l = this.statsCounter.activeUsage();
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public long pinnedUsage() {
        this.lock.lock();
        try {
            long l = this.statsCounter.pinnedUsage();
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public IRefCountedCacheStats stats() {
        this.lock.lock();
        try {
            IRefCountedCacheStats iRefCountedCacheStats = this.statsCounter.snapshot();
            return iRefCountedCacheStats;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void logCurrentState() {
        this.lock.lock();
        try {
            StringBuilder allFiles = new StringBuilder("\n");
            for (Map.Entry<K, Node<K, V>> entry : this.data.entrySet()) {
                String path = entry.getKey().toString();
                String file = path.substring(path.lastIndexOf(47));
                allFiles.append(file).append(" [RefCount: ").append(entry.getValue().refCount).append(" , Weight: ").append(entry.getValue().weight).append(" ]\n");
            }
            if (allFiles.length() > 1) {
                logger.trace(() -> "Cache entries : " + String.valueOf(allFiles));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void addNode(K key, boolean pinned, V value) {
        long weight = this.weigher.weightOf(value);
        Node<K, V> newNode = new Node<K, V>(key, value, weight);
        this.data.put(key, newNode);
        this.statsCounter.recordUsage(value, weight, pinned, false);
        this.incRef(key);
        this.evict();
    }

    private void replaceNode(Node<K, V> node, V newValue) {
        if (node.value != newValue) {
            Object oldValue = node.value;
            long oldWeight = node.weight;
            long newWeight = this.weigher.weightOf(newValue);
            node.value = newValue;
            node.weight = newWeight;
            this.statsCounter.recordReplacement(oldValue, newValue, oldWeight, newWeight, node.refCount > 0, node.pinned);
            this.listener.onRemoval(new RemovalNotification(node.key, oldValue, RemovalReason.REPLACED));
        }
        this.incRef(node.key);
        this.evict();
    }

    private void removeNode(K key) {
        Node<K, V> node = this.data.remove(key);
        if (node != null) {
            if (node.refCount > 0) {
                this.statsCounter.recordActiveUsage(node.value, node.weight, node.pinned, true);
            }
            if (node.evictable()) {
                this.lru.remove(node.key);
            }
            if (node.pinned) {
                this.statsCounter.recordPinnedUsage(node.value, node.weight, true);
            }
            this.statsCounter.recordRemoval(node.value, node.pinned, node.weight);
            this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.EXPLICIT));
        }
    }

    private boolean hasOverflowed() {
        return this.statsCounter.usage() >= this.capacity;
    }

    private void evict() {
        Iterator<Node<K, V>> iterator = this.lru.values().iterator();
        while (this.hasOverflowed() && iterator.hasNext()) {
            Node<K, V> node = iterator.next();
            iterator.remove();
            this.data.remove(node.key, node);
            this.statsCounter.recordEviction(node.value, node.weight);
            this.listener.onRemoval(new RemovalNotification(node.key, node.value, RemovalReason.CAPACITY));
        }
    }

    static class Node<K, V> {
        final K key;
        V value;
        long weight;
        int refCount;
        boolean pinned;

        Node(K key, V value, long weight) {
            this.key = key;
            this.value = value;
            this.weight = weight;
            this.refCount = 0;
            this.pinned = false;
        }

        public boolean evictable() {
            return this.refCount == 0 && !this.pinned;
        }
    }
}

