﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Tag.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 */
class CreateIntegrationRequest : public RDSRequest {
 public:
  AWS_RDS_API CreateIntegrationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateIntegration"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the database to use as the source for
   * replication.</p>
   */
  inline const Aws::String& GetSourceArn() const { return m_sourceArn; }
  inline bool SourceArnHasBeenSet() const { return m_sourceArnHasBeenSet; }
  template <typename SourceArnT = Aws::String>
  void SetSourceArn(SourceArnT&& value) {
    m_sourceArnHasBeenSet = true;
    m_sourceArn = std::forward<SourceArnT>(value);
  }
  template <typename SourceArnT = Aws::String>
  CreateIntegrationRequest& WithSourceArn(SourceArnT&& value) {
    SetSourceArn(std::forward<SourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the Redshift data warehouse to use as the target for
   * replication.</p>
   */
  inline const Aws::String& GetTargetArn() const { return m_targetArn; }
  inline bool TargetArnHasBeenSet() const { return m_targetArnHasBeenSet; }
  template <typename TargetArnT = Aws::String>
  void SetTargetArn(TargetArnT&& value) {
    m_targetArnHasBeenSet = true;
    m_targetArn = std::forward<TargetArnT>(value);
  }
  template <typename TargetArnT = Aws::String>
  CreateIntegrationRequest& WithTargetArn(TargetArnT&& value) {
    SetTargetArn(std::forward<TargetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the integration.</p>
   */
  inline const Aws::String& GetIntegrationName() const { return m_integrationName; }
  inline bool IntegrationNameHasBeenSet() const { return m_integrationNameHasBeenSet; }
  template <typename IntegrationNameT = Aws::String>
  void SetIntegrationName(IntegrationNameT&& value) {
    m_integrationNameHasBeenSet = true;
    m_integrationName = std::forward<IntegrationNameT>(value);
  }
  template <typename IntegrationNameT = Aws::String>
  CreateIntegrationRequest& WithIntegrationName(IntegrationNameT&& value) {
    SetIntegrationName(std::forward<IntegrationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services Key Management System (Amazon Web Services KMS) key
   * identifier for the key to use to encrypt the integration. If you don't specify
   * an encryption key, RDS uses a default Amazon Web Services owned key. </p>
   */
  inline const Aws::String& GetKMSKeyId() const { return m_kMSKeyId; }
  inline bool KMSKeyIdHasBeenSet() const { return m_kMSKeyIdHasBeenSet; }
  template <typename KMSKeyIdT = Aws::String>
  void SetKMSKeyId(KMSKeyIdT&& value) {
    m_kMSKeyIdHasBeenSet = true;
    m_kMSKeyId = std::forward<KMSKeyIdT>(value);
  }
  template <typename KMSKeyIdT = Aws::String>
  CreateIntegrationRequest& WithKMSKeyId(KMSKeyIdT&& value) {
    SetKMSKeyId(std::forward<KMSKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional set of non-secret key–value pairs that contains additional
   * contextual information about the data. For more information, see <a
   * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#encrypt_context">Encryption
   * context</a> in the <i>Amazon Web Services Key Management Service Developer
   * Guide</i>.</p> <p>You can only include this parameter if you specify the
   * <code>KMSKeyId</code> parameter.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAdditionalEncryptionContext() const { return m_additionalEncryptionContext; }
  inline bool AdditionalEncryptionContextHasBeenSet() const { return m_additionalEncryptionContextHasBeenSet; }
  template <typename AdditionalEncryptionContextT = Aws::Map<Aws::String, Aws::String>>
  void SetAdditionalEncryptionContext(AdditionalEncryptionContextT&& value) {
    m_additionalEncryptionContextHasBeenSet = true;
    m_additionalEncryptionContext = std::forward<AdditionalEncryptionContextT>(value);
  }
  template <typename AdditionalEncryptionContextT = Aws::Map<Aws::String, Aws::String>>
  CreateIntegrationRequest& WithAdditionalEncryptionContext(AdditionalEncryptionContextT&& value) {
    SetAdditionalEncryptionContext(std::forward<AdditionalEncryptionContextT>(value));
    return *this;
  }
  template <typename AdditionalEncryptionContextKeyT = Aws::String, typename AdditionalEncryptionContextValueT = Aws::String>
  CreateIntegrationRequest& AddAdditionalEncryptionContext(AdditionalEncryptionContextKeyT&& key,
                                                           AdditionalEncryptionContextValueT&& value) {
    m_additionalEncryptionContextHasBeenSet = true;
    m_additionalEncryptionContext.emplace(std::forward<AdditionalEncryptionContextKeyT>(key),
                                          std::forward<AdditionalEncryptionContextValueT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateIntegrationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateIntegrationRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Data filtering options for the integration. For more information, see <a
   * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/zero-etl.filtering.html">Data
   * filtering for Aurora zero-ETL integrations with Amazon Redshift</a> or <a
   * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/zero-etl.filtering.html">Data
   * filtering for Amazon RDS zero-ETL integrations with Amazon Redshift</a>. </p>
   */
  inline const Aws::String& GetDataFilter() const { return m_dataFilter; }
  inline bool DataFilterHasBeenSet() const { return m_dataFilterHasBeenSet; }
  template <typename DataFilterT = Aws::String>
  void SetDataFilter(DataFilterT&& value) {
    m_dataFilterHasBeenSet = true;
    m_dataFilter = std::forward<DataFilterT>(value);
  }
  template <typename DataFilterT = Aws::String>
  CreateIntegrationRequest& WithDataFilter(DataFilterT&& value) {
    SetDataFilter(std::forward<DataFilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the integration.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateIntegrationRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sourceArn;
  bool m_sourceArnHasBeenSet = false;

  Aws::String m_targetArn;
  bool m_targetArnHasBeenSet = false;

  Aws::String m_integrationName;
  bool m_integrationNameHasBeenSet = false;

  Aws::String m_kMSKeyId;
  bool m_kMSKeyIdHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_additionalEncryptionContext;
  bool m_additionalEncryptionContextHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_dataFilter;
  bool m_dataFilterHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
