﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/autoscaling-plans/AutoScalingPlans_EXPORTS.h>
#include <aws/autoscaling-plans/model/MetricDimension.h>
#include <aws/autoscaling-plans/model/MetricStatistic.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace AutoScalingPlans {
namespace Model {

/**
 * <p>Represents a CloudWatch metric of your choosing that can be used for dynamic
 * scaling as part of a target tracking scaling policy. </p> <p>To create your
 * customized scaling metric specification:</p> <ul> <li> <p>Add values for each
 * required parameter from CloudWatch. You can use an existing metric, or a new
 * metric that you create. To use your own metric, you must first publish the
 * metric to CloudWatch. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publish
 * Custom Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.</p> </li> <li>
 * <p>Choose a metric that changes proportionally with capacity. The value of the
 * metric should increase or decrease in inverse proportion to the number of
 * capacity units. That is, the value of the metric should decrease when capacity
 * increases. </p> </li> </ul> <p>For information about terminology, available
 * metrics, or how to publish new metrics, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html">Amazon
 * CloudWatch Concepts</a> in the <i>Amazon CloudWatch User Guide</i>.
 * </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-plans-2018-01-06/CustomizedScalingMetricSpecification">AWS
 * API Reference</a></p>
 */
class CustomizedScalingMetricSpecification {
 public:
  AWS_AUTOSCALINGPLANS_API CustomizedScalingMetricSpecification() = default;
  AWS_AUTOSCALINGPLANS_API CustomizedScalingMetricSpecification(Aws::Utils::Json::JsonView jsonValue);
  AWS_AUTOSCALINGPLANS_API CustomizedScalingMetricSpecification& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_AUTOSCALINGPLANS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the metric.</p>
   */
  inline const Aws::String& GetMetricName() const { return m_metricName; }
  inline bool MetricNameHasBeenSet() const { return m_metricNameHasBeenSet; }
  template <typename MetricNameT = Aws::String>
  void SetMetricName(MetricNameT&& value) {
    m_metricNameHasBeenSet = true;
    m_metricName = std::forward<MetricNameT>(value);
  }
  template <typename MetricNameT = Aws::String>
  CustomizedScalingMetricSpecification& WithMetricName(MetricNameT&& value) {
    SetMetricName(std::forward<MetricNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace of the metric.</p>
   */
  inline const Aws::String& GetNamespace() const { return m_namespace; }
  inline bool NamespaceHasBeenSet() const { return m_namespaceHasBeenSet; }
  template <typename NamespaceT = Aws::String>
  void SetNamespace(NamespaceT&& value) {
    m_namespaceHasBeenSet = true;
    m_namespace = std::forward<NamespaceT>(value);
  }
  template <typename NamespaceT = Aws::String>
  CustomizedScalingMetricSpecification& WithNamespace(NamespaceT&& value) {
    SetNamespace(std::forward<NamespaceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The dimensions of the metric.</p> <p>Conditional: If you published your
   * metric with dimensions, you must specify the same dimensions in your customized
   * scaling metric specification.</p>
   */
  inline const Aws::Vector<MetricDimension>& GetDimensions() const { return m_dimensions; }
  inline bool DimensionsHasBeenSet() const { return m_dimensionsHasBeenSet; }
  template <typename DimensionsT = Aws::Vector<MetricDimension>>
  void SetDimensions(DimensionsT&& value) {
    m_dimensionsHasBeenSet = true;
    m_dimensions = std::forward<DimensionsT>(value);
  }
  template <typename DimensionsT = Aws::Vector<MetricDimension>>
  CustomizedScalingMetricSpecification& WithDimensions(DimensionsT&& value) {
    SetDimensions(std::forward<DimensionsT>(value));
    return *this;
  }
  template <typename DimensionsT = MetricDimension>
  CustomizedScalingMetricSpecification& AddDimensions(DimensionsT&& value) {
    m_dimensionsHasBeenSet = true;
    m_dimensions.emplace_back(std::forward<DimensionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The statistic of the metric.</p>
   */
  inline MetricStatistic GetStatistic() const { return m_statistic; }
  inline bool StatisticHasBeenSet() const { return m_statisticHasBeenSet; }
  inline void SetStatistic(MetricStatistic value) {
    m_statisticHasBeenSet = true;
    m_statistic = value;
  }
  inline CustomizedScalingMetricSpecification& WithStatistic(MetricStatistic value) {
    SetStatistic(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unit of the metric. </p>
   */
  inline const Aws::String& GetUnit() const { return m_unit; }
  inline bool UnitHasBeenSet() const { return m_unitHasBeenSet; }
  template <typename UnitT = Aws::String>
  void SetUnit(UnitT&& value) {
    m_unitHasBeenSet = true;
    m_unit = std::forward<UnitT>(value);
  }
  template <typename UnitT = Aws::String>
  CustomizedScalingMetricSpecification& WithUnit(UnitT&& value) {
    SetUnit(std::forward<UnitT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_metricName;

  Aws::String m_namespace;

  Aws::Vector<MetricDimension> m_dimensions;

  MetricStatistic m_statistic{MetricStatistic::NOT_SET};

  Aws::String m_unit;
  bool m_metricNameHasBeenSet = false;
  bool m_namespaceHasBeenSet = false;
  bool m_dimensionsHasBeenSet = false;
  bool m_statisticHasBeenSet = false;
  bool m_unitHasBeenSet = false;
};

}  // namespace Model
}  // namespace AutoScalingPlans
}  // namespace Aws
