"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PermissionControlledUiSettingsWrapper = void 0;
var _server = require("../../../server");
var _utils = require("../../../server/utils");
var _utils2 = require("../utils");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
/**
 * Wrapper for admin UI settings that enforces permission controls
 * Handles special cases for admin UI settings with appropriate access controls
 */
class PermissionControlledUiSettingsWrapper {
  /**
   * @param isPermissionControlEnabled
   */
  constructor(isPermissionControlEnabled) {
    this.isPermissionControlEnabled = isPermissionControlEnabled;
    _defineProperty(this, "aclInstance", void 0);
    _defineProperty(this, "wrapperFactory", wrapperOptions => {
      const createUiSettingWithPermission = async (type, attributes, options = {}) => {
        if (type === 'config' && options.id && options.id === _utils2.DASHBOARD_ADMIN_SETTINGS_ID) {
          return this.createPermissionUiSetting(attributes, options, wrapperOptions);
        }
        return wrapperOptions.client.create(type, attributes, options);
      };
      const updateUiSettingsWithPermission = async (type, id, attributes, options = {}) => {
        if (type === 'config' && id === _utils2.DASHBOARD_ADMIN_SETTINGS_ID) {
          try {
            return await wrapperOptions.client.update(type, _utils2.DASHBOARD_ADMIN_SETTINGS_ID, attributes, options);
          } catch (error) {
            if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
              return this.createPermissionUiSetting(attributes, options, wrapperOptions);
            } else {
              throw error;
            }
          }
        }
        return wrapperOptions.client.update(type, id, attributes, options);
      };
      const bulkCreateWithPermission = async (objects, options) => {
        if ((options === null || options === void 0 ? void 0 : options.id) === _utils2.DASHBOARD_ADMIN_SETTINGS_ID) {
          throw new Error('Bulk create is not supported for admin settings');
        }

        // Prevent bulk creation if any object targets admin settings
        const adminSettingObject = objects.find(obj => obj.type === 'config' && obj.id === _utils2.DASHBOARD_ADMIN_SETTINGS_ID);
        if (adminSettingObject) {
          throw new Error('Bulk create is not supported for admin settings');
        }
        return wrapperOptions.client.bulkCreate(objects, options);
      };
      const bulkUpdateWithPermission = async (objects, options) => {
        const adminSettingObject = objects.find(obj => obj.type === 'config' && obj.id === _utils2.DASHBOARD_ADMIN_SETTINGS_ID);
        if (adminSettingObject) {
          throw new Error('Bulk update is not supported for admin settings');
        }
        return wrapperOptions.client.bulkUpdate(objects, options);
      };
      return {
        ...wrapperOptions.client,
        create: createUiSettingWithPermission,
        bulkCreate: bulkCreateWithPermission,
        delete: wrapperOptions.client.delete,
        update: updateUiSettingsWithPermission,
        bulkUpdate: bulkUpdateWithPermission,
        get: wrapperOptions.client.get,
        checkConflicts: wrapperOptions.client.checkConflicts,
        errors: wrapperOptions.client.errors,
        addToNamespaces: wrapperOptions.client.addToNamespaces,
        deleteFromNamespaces: wrapperOptions.client.deleteFromNamespaces,
        find: wrapperOptions.client.find,
        bulkGet: wrapperOptions.client.bulkGet,
        deleteByWorkspace: wrapperOptions.client.deleteByWorkspace
      };
    });
  }
  checkUiSettingsWritePermission(wrapperOptions) {
    // If saved object permission is disabled, everyone should be treated as admin here
    const isDashboardAdmin = (0, _utils.getWorkspaceState)(wrapperOptions.request).isDashboardAdmin !== false || !this.isPermissionControlEnabled;
    if (!isDashboardAdmin) {
      throw _server.SavedObjectsErrorHelpers.decorateForbiddenError(new Error('No permission for admin UI settings operations'));
    }
  }
  getAclInstance() {
    if (!this.isPermissionControlEnabled) {
      return undefined;
    }
    if (!this.aclInstance) {
      // Allow read for all users but no write for any users
      // This means only dashboard admin would have permission to bypass ACL
      this.aclInstance = new _server.ACL().addPermission(['read'], {
        users: ['*']
      });
    }
    return this.aclInstance;
  }
  async createPermissionUiSetting(attributes, options, wrapperOptions) {
    this.checkUiSettingsWritePermission(wrapperOptions);
    const aclInstance = this.getAclInstance();
    return wrapperOptions.client.create('config', attributes, {
      ...options,
      overwrite: true,
      id: _utils2.DASHBOARD_ADMIN_SETTINGS_ID,
      ...(aclInstance ? {
        permissions: aclInstance.getPermissions()
      } : {})
    });
  }
}
exports.PermissionControlledUiSettingsWrapper = PermissionControlledUiSettingsWrapper;